/* MOLESKINE MOLESKINE MOLESKINE NOT EMACS */


#include "gpgme_cpp.h"
#include <cstdio>       // for snprintf
#include <algorithm>
#include <iostream>
#include <iomanip>

using std::cerr;
using std::endl;
using std::hex;
using std::sort;

namespace gpgme
{
    void errswitch(GpgmeError err)
    {
        switch (err)
        {
            case GPGME_EOF:             throw EndOfFile();
            case GPGME_No_Error:        break;
            case GPGME_General_Error:   throw GeneralError();
            case GPGME_Out_Of_Core:     throw OutOfCore();
            case GPGME_Invalid_Value:   throw InvalidValue();
            case GPGME_Busy:            throw Busy();
            case GPGME_No_Request:      throw NoRequest();
            case GPGME_Exec_Error:      throw ExecError();
            case GPGME_Too_Many_Procs:  throw TooManyProcs();
            case GPGME_Pipe_Error:      throw PipeError();
            case GPGME_No_Recipients:   throw NoRecipients();
            case GPGME_Invalid_Recipients: 
                                        throw InvalidRecipients();
            case GPGME_No_Data:         throw NoData();
            case GPGME_Conflict:        throw Conflict();
            case GPGME_Not_Implemented: throw NotImplemented();
            case GPGME_Read_Error:      throw ReadError();
            case GPGME_Write_Error:     throw WriteError();
            case GPGME_Invalid_Type:    throw InvalidType();
            case GPGME_Invalid_Mode:    throw InvalidMode();
            case GPGME_File_Error:      throw FileError();
            case GPGME_Decryption_Failed: 
                                        throw DecryptionFailed();
            case GPGME_No_Passphrase:   throw NoPassphrase();
            case GPGME_Canceled:        throw Canceled();
            case GPGME_Invalid_Key:     throw InvalidKey();
            case GPGME_Invalid_Engine:  throw InvalidEngine();
            default:                    throw GPGMEerr("unspecified");
        }
    }

    Context::Context(Protocol proto)
    {
        errswitch(gpgme_new(&_ctx));
        setProtocol(proto);
        setKeylistMode(LOCAL);
        _ctxNeedsFreeing = true;
    }

    Context::Context(const Context& original)
    {
        // This copy constructor gives the new object the same
        // protocol and keylist-mode as the original, but creates
        // a new GpgmeCtx context.  This operation is computationally
        // expensive.
        errswitch(gpgme_new(&_ctx));
        setProtocol(original._currentProto);
        setKeylistMode(original._currentKeyringLocale);
        _ctxNeedsFreeing = true;
    }

    Context& Context::operator=(const Context& original)
    {
        // if we're assigning two Contexts equal to each other,
        // only the first Context gets to free the GpgmeCtx.  The
        // flipside: whenever the first Context goes out of scope,
        // this second one will become dangerous (will refer to
        // memory which has been returned to the pool).
        //
        // Please note that this assignment is computationally
        // cheap.  Whenever possible, use it over a constructor.
        _ctx = original._ctx;
        _ctxNeedsFreeing = false;
        return *this;
    }

    Context::~Context()
    {
        if (_ctxNeedsFreeing) gpgme_release(_ctx);
    }

    void Context::setProtocol(Protocol proto)
    {
        _currentProto = proto;
        switch (_currentProto)
        {
            case OpenPGP:
                errswitch(gpgme_set_protocol(_ctx, 
                    GPGME_PROTOCOL_OpenPGP));
                break;
            case CMS:
                errswitch(gpgme_set_protocol(_ctx, 
                    GPGME_PROTOCOL_CMS));
                break;
        }
    }

    const bool Context::getArmor() const throw()
    {
        if (gpgme_get_armor(_ctx)) return true;
        return false;
    }

    void Context::setArmor(bool armor) throw()
    {
        switch (armor)
        {
            case true:
                gpgme_set_armor(_ctx, 1);
                break;
            case false:
                gpgme_set_armor(_ctx, 0);
                break;
        }
    }

    const bool Context::getTextmode() const throw()
    {
        if (gpgme_get_textmode(_ctx)) return true;
        return false;
    }

    void Context::setTextmode(bool which) throw()
    {
        switch (which)
        {
            case true:
                gpgme_set_textmode(_ctx, 1);
                break;
            case false:
                gpgme_set_textmode(_ctx, 0);
                break;
        }
    }

    const int Context::getIncludeCerts() const throw()
    {
        if (_currentProto == OpenPGP) return -127;
        return gpgme_get_include_certs(_ctx);
    }

    void Context::setIncludeCerts(int certs) throw()
    {
        if (_currentProto == OpenPGP) return;
        gpgme_set_include_certs(_ctx, certs);
    }

    const KeyringLocale Context::getKeylistMode() const
    {
        switch (gpgme_get_keylist_mode(_ctx))
        {
            case 0:
                throw GPGMEerr("Invalid context");
                break;
            case GPGME_KEYLIST_MODE_LOCAL:
                return LOCAL;
            case GPGME_KEYLIST_MODE_EXTERN:
                return EXTERN;
            default:
                throw GPGMEerr("Nonspecific GPGME error");
                break;
        }
    }

    void Context::setKeylistMode(KeyringLocale locale) throw()
    {
        _currentKeyringLocale = locale;
        switch (locale)
        {
            case LOCAL:
                gpgme_set_keylist_mode(_ctx, GPGME_KEYLIST_MODE_LOCAL);
                break;
            case EXTERN:
                gpgme_set_keylist_mode(_ctx, GPGME_KEYLIST_MODE_EXTERN);
                break;
        }
    }

    namespace keys
    {
        UID::UID(GpgmeKey key, unsigned long index)
        {
            unsigned long id = 
                gpgme_key_get_ulong_attr(key, GPGME_ATTR_USERID,
                    NULL, index);
            const char* name =
                gpgme_key_get_string_attr(key, GPGME_ATTR_NAME,
                    NULL, index);
            const char* email =
                gpgme_key_get_string_attr(key, GPGME_ATTR_EMAIL,
                    NULL, index);
            const char* comment =
                gpgme_key_get_string_attr(key, GPGME_ATTR_COMMENT,
                    NULL, index);
            const char* validity =
                gpgme_key_get_string_attr(key, GPGME_ATTR_VALIDITY,
                    NULL, index);
            bool uidr = 
                gpgme_key_get_ulong_attr(key, GPGME_ATTR_UID_REVOKED,
                    NULL, index) ? true : false;
            bool uidi = 
                gpgme_key_get_ulong_attr(key, GPGME_ATTR_UID_INVALID,
                    NULL, index) ? true : false;
            
            if (name == 0) throw GPGMEerr("Bad UID");
            
            _index = index;
            _id = id;
            _name = string(name);
            _email = string(email);
            _comment = string(comment);
            _validity = string(validity);
            _revoked = uidr;
            if (_revoked) _name = string("REVOKED ") + _name;
            _invalid = uidi;
            if (_invalid) _name = string("INVALID ") + _name;
        }
        
        const unsigned long UID::id() const throw()
        {
            return _id;
        }
        
        const string& UID::name() const throw()
        {
            return _name;
        }
        
        const string& UID::email() const throw()
        {
            return _email;
        }
        
        const string& UID::comment() const throw()
        {
            return _comment;
        }
        
        const string& UID::validity() const throw()
        {
            return _validity;
        }
        
        const bool UID::revoked() const throw()
        {
            return _revoked;
        }
        
        const bool UID::invalid() const throw()
        {
            return _invalid;
        }
        
        const bool UID::trusted() const throw()
        {
            if (_revoked || _invalid) return false;
            if (_validity=="f" || _validity=="u") return true;
            return false;
        }


        Key::Key()
        {
            _name = _id = _fingerprint = _algo = _email = _comment =
                _validity = _capabilities = "NONE";
                
            _isSecret = false;
            _keyRevoked = _keyInvalid = _keyExpired = _keyDisabled =
                _useridRevoked = _useridInvalid = true;
            _created = _expires = _length = _userid = 0;
            _key = NULL;
        }
        
        Key::Key(GpgmeKey keydata) : _key(keydata)
        {
            // string entities
            
            const char *name    = gpgme_key_get_string_attr(_key,
                GPGME_ATTR_NAME, NULL, 0);
            _id                 = gpgme_key_get_string_attr(_key, 
                GPGME_ATTR_KEYID, NULL, 0);
            _fingerprint        = gpgme_key_get_string_attr(_key, 
                GPGME_ATTR_FPR, NULL, 0);
            _algo               = gpgme_key_get_string_attr(_key, 
                GPGME_ATTR_ALGO, NULL, 0);
            const char* email   = gpgme_key_get_string_attr(_key, 
                GPGME_ATTR_EMAIL, NULL, 0);
            const char *comment = gpgme_key_get_string_attr(_key, 
                GPGME_ATTR_COMMENT, NULL, 0);
            const char *validity= gpgme_key_get_string_attr(_key, 
                GPGME_ATTR_VALIDITY, NULL, 0);
            _capabilities       = gpgme_key_get_string_attr(_key, 
                GPGME_ATTR_KEY_CAPS, NULL, 0);
            
            name        ? _name  = name  : _name  = "Unknown";
            email       ? _email = email : _email = "Unknown";
            comment     ? _comment = comment : _comment = "";
            validity    ? _validity = validity : _validity = "NONE";
            
            try
            {
                unsigned long i = 0;
                while (1)
                {
                    UID uid(_key, i++);
                    _uids.push_back(uid);
                }
            }
            catch (GPGMEerr &e)
            {
            }
            
            // numeric entities
            _created = gpgme_key_get_ulong_attr(_key, 
                GPGME_ATTR_CREATED, NULL, 0);
            _expires = gpgme_key_get_ulong_attr(_key, 
                GPGME_ATTR_EXPIRE, NULL, 0);
            _isSecret = gpgme_key_get_ulong_attr(_key, 
                GPGME_ATTR_IS_SECRET, NULL, 0) ? true : false;
            _keyRevoked = gpgme_key_get_ulong_attr(_key, 
                GPGME_ATTR_KEY_REVOKED, NULL, 0) ? true : false;
            _keyInvalid = gpgme_key_get_ulong_attr(_key, 
                GPGME_ATTR_KEY_INVALID, NULL, 0) ? true : false;
            _keyExpired = gpgme_key_get_ulong_attr(_key, 
                GPGME_ATTR_KEY_EXPIRED, NULL, 0) ? true : false;
            _keyDisabled = gpgme_key_get_ulong_attr(_key, 
                GPGME_ATTR_KEY_DISABLED, NULL, 0) ? true : false;
            _length = gpgme_key_get_ulong_attr(_key, 
                GPGME_ATTR_LEN, NULL, 0);
            _userid = gpgme_key_get_ulong_attr(_key, 
                GPGME_ATTR_USERID, NULL, 0);
            _useridRevoked = gpgme_key_get_ulong_attr(_key, 
                GPGME_ATTR_USERID, NULL, 0) ? true : false;
            _useridInvalid = gpgme_key_get_ulong_attr(_key, 
                GPGME_ATTR_USERID, NULL, 0) ? true : false;
        }
        
        Key::Key(const Key& data)
        {
            _key = data.getGpgmeKey();
            if (_key == 0) return;
            gpgme_key_ref(_key);
            _id = data.id();
            _fingerprint = data.fingerprint();
            _algo = data.algo();
            _name = data.name();
            _email = data.email();
            _comment = data.comment();
            _validity = data.validity();
            _capabilities = data.capabilities();
            
            _created = data.created();
            _expires = data.expires();
            _isSecret = data.isSecret();
            _keyRevoked = data.keyRevoked();
            _keyInvalid = data.keyInvalid();
            _keyExpired = data.keyExpired();
            _keyDisabled = data.keyDisabled();
            _length = data.length();
            _userid = data.userid();
            _useridRevoked = data.useridRevoked();
            _useridInvalid = data.useridInvalid();
            
            _uids.clear();
            
            Key::const_iterator iter = data.uids().begin();
            while (iter != data.uids().end())
                _uids.push_back(*iter++);
        }
        
        Key& Key::operator=(const Key& data)
        {
            _key = data.getGpgmeKey();
            if (!(_key)) return *this;
            gpgme_key_ref(_key);
            _id = data.id();
            _fingerprint = data.fingerprint();
            _algo = data.algo();
            _name = data.name();
            _email = data.email();
            _comment = data.comment();
            _validity = data.validity();
            _capabilities = data.capabilities();
            
            _created = data.created();
            _expires = data.expires();
            _isSecret = data.isSecret();
            _keyRevoked = data.keyRevoked();
            _keyInvalid = data.keyInvalid();
            _keyExpired = data.keyExpired();
            _keyDisabled = data.keyDisabled();
            _length = data.length();
            _userid = data.userid();
            _useridRevoked = data.useridRevoked();
            _useridInvalid = data.useridInvalid();
            
            _uids.clear();
            
            Key::const_iterator iter = data.uids().begin();
            while (iter != data.uids().end())
                _uids.push_back(*iter++);
            return *this;
        }
        
        Key::~Key()
        {
            if (_key) gpgme_key_unref(_key);
        }

        string Key::getAsXML() const throw()
        {
            char *foo = gpgme_key_get_as_xml(_key);
            string bar(foo);
            delete foo;
            return bar;
        }
        
        bool Key::operator<(const Key &right) const throw()
        {
            return (this->_id < right.id());
        }
        
        bool Key::operator<(const string &right) const throw()
        {
            return (this->_id < right);
        }
        
        bool Key::operator<(const char* right) const throw()
        {
            return (this->_id < string(right));
        }
        
        bool Key::operator==(const Key &right) const throw()
        {
            return (this->_id == right.id());
        }
        
        bool Key::operator==(const string &right) const throw()
        {
            return (this->_id == right);
        }
        
        bool Key::operator==(const char *right) const throw()
        {
            return (this->_id == string(right));
        }
        
        bool Key::operator>(const Key &right) const throw()
        {
            return (this->_id > right.id());
        }
        
        bool Key::operator>(const string &right) const throw()
        {
            return (this->_id > right);
        }
        
        bool Key::operator>(const char *right) const throw()
        {
            return (this->_id > string(right));
        }
        
        GpgmeKey Key::getGpgmeKey() const throw()
        {
            return _key;
        }
        
        const string& Key::id() const throw()
        {
            return _id;
        }
        
        const string& Key::fingerprint() const throw()
        {
            return _fingerprint;
        }
        
        const string& Key::algo() const throw()
        {
            return _algo;
        }
        
        const string& Key::name() const throw()
        {
            return _name;
        }
        
        const string& Key::email() const throw()
        {
            return _email;
        }
        
        const string& Key::comment() const throw()
        {
            return _comment;
        }
        
        const string& Key::validity() const throw()
        {
            return _validity;
        }
        
        const string& Key::capabilities() const throw()
        {
            return _capabilities;
        }
        
        const unsigned long Key::created() const throw()
        {
            return _created;
        }
        
        const unsigned long Key::expires() const throw()
        {
            return _expires;
        }
        
        const bool Key::isSecret() const throw()
        {
            return _isSecret;
        }
        
        const bool Key::keyRevoked() const throw()
        {
            return _keyRevoked;
        }
        
        const bool Key::keyInvalid() const throw()
        {
            return _keyInvalid;
        }
        
        const bool Key::keyExpired() const throw()
        {
            return _keyExpired;
        }
        
        const bool Key::keyDisabled() const throw()
        {
            return _keyDisabled;
        }
        
        const unsigned long Key::length() const throw()
        {
            return _length;
        }
        
        const unsigned long Key::userid() const throw()
        {
            return _userid;
        }
        
        const bool Key::useridRevoked() const throw()
        {
            return _useridRevoked;
        }
        
        const bool Key::useridInvalid() const throw()
        {
            return _useridInvalid;
        }
        
        const vector<UID>& Key::uids() const throw()
        {
            return _uids;
        }
        
        Key::const_iterator Key::begin() const throw()
        {
            return _uids.begin();
        }
        
        Key::const_iterator Key::end() const throw()
        {
            return _uids.end();
        }
        
        
        
        KeyDB::KeyDB(KeyringLocale where,
            bool privatekeysonly)
            : _privateonly(privatekeysonly)
        {
            setKeylistMode(where);
            update();
        }
        
        KeyDB::~KeyDB()
        {
            _keyring.clear();
        }
        
        void KeyDB::update()
        {
            GpgmeKey key;
            _keyring.clear();
            try
            {
                switch (_privateonly)
                {
                    case false:
                        errswitch(gpgme_op_keylist_start(_ctx, NULL, 0));
                        break;
                    case true:
                        errswitch(gpgme_op_keylist_start(_ctx, NULL, 1));
                        break;
                }
                while (1)
                {
                    errswitch(gpgme_op_keylist_next(_ctx, &key));
                    Key tempkey(key);
                    _keyring[tempkey.id()] = tempkey;
                }
            }
            catch (EndOfFile &e)
            {
                // empty -- we're expecting it, after all
            }
        }
                
        // For now, generateKey will only generate DSA keys with
        // ELG-E subkeys.  Nothing else, nada, zero, zilch.  RSA
        // support is on the TODO, along with a few million other
        // things.
        void KeyDB::generateKey(KeyType kt, unsigned int klen,
                SubkeyType skt, unsigned int sklen, string name,
                string email, string comment, string passphrase)
        {
            string keytype = "DSA";
            string subkeytype = "ELG-E";
            string nl = "\n";
            if (keytype == "DSA") klen = 1024;
            
            char buf1[] = { 0, 0, 0, 0, 0, 0, 0, 0 };
            char buf2[] = { 0, 0, 0, 0, 0, 0, 0, 0 };
            
            snprintf(buf1, 7, "%d", klen);
            snprintf(buf2, 7, "%d", sklen);
            
            string string_klen(buf1);
            string string_sklen(buf2);
            
            string parms =
                string("<GnupgKeyParms format=\"internal\">\n") +
                string("Key-Type: ") + keytype + nl +
                string("Key-Length: ") + string_klen + nl +
                string("Subkey-Type: ") + subkeytype + nl +
                string("Subkey-Length: ") + string_sklen + nl +
                string("Name-Real: ") + name + nl +
                string("Name-Comment: ") + comment + nl +
                string("Name-Email: ") + email + nl +
                string("Expire-Date: 0\n") +
                string("Passphrase: ") + passphrase + nl +
                string("</GnupgKeyParms>");
            
            errswitch(gpgme_op_genkey(_ctx, parms.c_str(), 
                NULL, NULL));
            
            update();
        }        
        
        unsigned long KeyDB::size() const throw()
        {
            return _keyring.size();
        }
        
        const Key& KeyDB::operator[](const string& index)
        {
            return _keyring[index];
        }
        
        map<string, gpgme::keys::Key>::const_iterator 
        KeyDB::begin() const throw()
        {
            return _keyring.begin();
        }
        
        map<string, gpgme::keys::Key>::const_iterator 
        KeyDB::end() const throw()
        {
            return _keyring.end();
        }
    }
    
    
    namespace util
    {
        GpgmeRecipients Recipients::makeGpgmeRecipients() const
        {
            GpgmeRecipients recips;
            errswitch(gpgme_recipients_new(&recips));
            const_iterator first = _recips.begin();
            const_iterator last = _recips.end();
            while (first != last)
            {
                errswitch(gpgme_recipients_add_name(recips,
                    first->id().c_str()));
                first++;
            }
            return recips;
        }
        
        GpgmeData Data::makeGpgmeData() const
        {
            GpgmeData data;
            // This appears to be an ugly hack, but isn't.  According to
            // the latest spec, memory in a vector is guaranteed to be
            // contiguous--hence, we can do the following without fear of
            // segfault.
            //
            // It's _still_ butt-ugly.  Though efficient.
            //
            if (_data.size() > 0)
                errswitch(gpgme_data_new_from_mem(&data, &(_data[0]), 
                    _data.size(), 1));
            else
                errswitch(gpgme_data_new(&data));
            return data;
        }
    }
        
    void Engine::updateSigners()
    {
        gpgme_signers_clear(_ctx);
        list<keys::Key>::const_iterator iter = _signers.begin();
        while (iter != _signers.end())
        {
            gpgme_signers_add(_ctx, iter->getGpgmeKey());
            iter++;
        }
    }
    
    gpgme::util::Data Engine::encrypt(const util::Data &plaintext)
    {
        if (_recip.size() == 0) throw GPGMEerr("No Recipients");
        
        GpgmeData _plaintext = plaintext.makeGpgmeData();
        GpgmeData _ciphertext;
        errswitch(gpgme_data_new(&_ciphertext));
        GpgmeRecipients _recipients = _recip.makeGpgmeRecipients();

        errswitch(gpgme_op_encrypt(getGpgmeCtx(), 
            _recipients, _plaintext, _ciphertext));
        
        // _ct gets released in the Data constructor
        util::Data retval(_ciphertext);
        gpgme_data_release(_plaintext);
        gpgme_recipients_release(_recipients);
        
        return retval;
    }
    
    gpgme::util::Data Engine::decrypt(const util::Data &ciphertext)
    {
        // this is only here temporarily
        if (_callback == NULL) throw GPGMEerr("No callback function");
        GpgmeData _ciphertext = ciphertext.makeGpgmeData();
        GpgmeData _plaintext;
        errswitch(gpgme_data_new(&_plaintext));
        errswitch(gpgme_op_decrypt(getGpgmeCtx(), 
            _ciphertext, _plaintext));
        util::Data retval(_plaintext);
        gpgme_data_release(_ciphertext);
        return retval;
    }
    
    GpgmeSigStat Engine::verify(const util::Data &signature,
        const util::Data &data)
    {
        GpgmeData _signature = signature.makeGpgmeData();
        GpgmeData _data;
        if (&data) _data = data.makeGpgmeData();
        GpgmeSigStat retval;
        errswitch(gpgme_op_verify(getGpgmeCtx(), _signature,
            _data, &retval));
        return retval;
    }
    
    pair<util::Data, GpgmeSigStat> Engine::decrypt_verify
        (const util::Data &ciphertext)
    {
        GpgmeData _ciphertext = ciphertext.makeGpgmeData();
        GpgmeData _plaintext;
        GpgmeSigStat sigstat;

        gpgme_data_new(&_plaintext);

        errswitch(gpgme_op_decrypt_verify(getGpgmeCtx(),
            _ciphertext, _plaintext, &sigstat));
        
        return pair<util::Data, GpgmeSigStat>(_plaintext, sigstat);
    }
    
    util::Data Engine::encrypt_sign(const util::Data &plaintext)
    {
        if (_recip.size() == 0) throw GPGMEerr("No Recipients");
        if (_signers.size() == 0) throw GPGMEerr("No Signers");
        updateSigners();
        
        GpgmeData _plaintext = plaintext.makeGpgmeData();
        GpgmeData _ciphertext;
        errswitch(gpgme_data_new(&_ciphertext));
        GpgmeRecipients _recipients = _recip.makeGpgmeRecipients();

        errswitch(gpgme_op_encrypt_sign(getGpgmeCtx(), 
            _recipients, _plaintext, _ciphertext));
        
        // _ct gets released in the Data constructor
        util::Data retval(_ciphertext);
        gpgme_data_release(_plaintext);
        gpgme_recipients_release(_recipients);
        
        return retval;       
    }
    
    std::ostream& operator<<(std::ostream &os, util::Data &z)
    {
        util::Data::iterator first, last;
        first = z.begin(); last = z.end();
        while (first != last) os << *first++;
        return os;
    }
}

